//
// $Id: fmsound.cc,v 1.16 2001/07/18 16:24:41 nishi Exp $
//
// Copyright (C) 2001 Shouhei Nishi.
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions
// are met:
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
// 3. The name of author may not be used to endorse or promote products
//    derived from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
// OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
// LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
// OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE.
//





#include "bochs.h"
#if BX_EMULATION_TOWNS
#include "mamewrap.h"
extern "C" {
#include "2612intf.h"
#include "rf5c68.h"
#include "sndintrf.h"
#include "timer.h"
}
#define LOG_THIS bx_fmsound.


bx_fmsound_c bx_fmsound;
#if BX_USE_FMSND_SMF
#define this (&bx_fmsound)
#endif


bx_fmsound_c::bx_fmsound_c(void)
{
}

bx_fmsound_c::~bx_fmsound_c(void)
{
  // nothing for now
}


static struct YM2612interface ym2612_interface =
{
        1,                      /* 1 chip */
        53693100 / 7,
        { 0x7fffffff },
        { 0 },
        { 0 },
        { 0 },
        { 0 },
        { bx_fmsound_c::ym2612_irq_handler },
};

static struct RF5C68interface rf5c68_interface = {
  2150000,
  100
};

static struct MachineDriver fm_towns =
{
  SOUND_SUPPORTS_STEREO,
  60.0,
  {
    {
      SOUND_YM2612,
      &ym2612_interface
    },
    {
      SOUND_RF5C68,
      &rf5c68_interface
    }
  }
};

struct RunningMachine a_Machine={
  40000,
  &fm_towns,
  NULL
};
struct RunningMachine *Machine=&a_Machine;

  void
bx_fmsound_c::init(bx_devices_c *d)
{
  // FM-TOWNS SOUND

  BX_FMSOUND_THIS devices = d;

  int i;

  static struct {
    Bit16u num;
    Boolean read;
    Boolean write;
  } fmsound_ports[]={
    {0x04d5,1,1},
    {0x04d8,1,1},
    {0x04da,1,1},
    {0x04dc,1,1},
    {0x04de,1,1},
    {0x04e0,1,1},
    {0x04e1,1,1},
    {0x04e2,1,1},
    {0x04e3,1,1},
    {0x04e7,1,0},
    {0x04e8,1,1},
    {0x04e9,1,0},
    {0x04ea,1,1},
    {0x04eb,1,0},
    {0x04ec,1,1},
    {0x04f0,0,1},
    {0x04f1,0,1},
    {0x04f2,0,1},
    {0x04f3,0,1},
    {0x04f4,0,1},
    {0x04f5,0,1},
    {0x04f6,0,1},
    {0x04f7,0,1},
    {0x04f8,0,1},
    {0,0,0}
  };

  i=0;
  while(fmsound_ports[i].num!=0) {
    if(fmsound_ports[i].read) {
      BX_FMSOUND_THIS devices->register_io_read_handler(this, read_handler,
							 fmsound_ports[i].num,
							 "FM-TOWNS SOUND");
    }
    if(fmsound_ports[i].write) {
      BX_FMSOUND_THIS devices->register_io_write_handler(this, write_handler,
							  fmsound_ports[i].num,
							  "FM-TOWNS SOUND");
    }
    i++;
  }

  timer_init();

  sound_start();

  BX_FMSOUND_THIS pcm_int_mask =
    BX_FMSOUND_THIS pcm_int_flag = 0;
  timer_pulse(1/60.0,0,(void(*)(int))sound_update);
}
  // static IO port read callback handler
  // redirects to non-static class handler to avoid virtual functions

  Bit32u
bx_fmsound_c::read_handler(void *this_ptr, Bit32u address, unsigned io_len)
{
#if !BX_USE_FMSND_SMF
  bx_fmsound_c *class_ptr = (bx_fmsound_c *) this_ptr;

  return( class_ptr->read(address, io_len) );
}

  Bit32u
bx_fmsound_c::read(Bit32u address, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif
  Bit32u ret;
#define RETURN(x) do { ret = (x); goto read_return; } while (0)

  if (io_len > 1)
    BX_PANIC(("fmsound: io read from address %08x len=%u\n",
	      (unsigned) address, (unsigned) io_len));

  switch (address) {
    case 0x04d8:
    case 0x04da: // Work Around for SimCity2000
    case 0x04dc: // Work Around for SimCity2000
    case 0x04de: // Work Around for SimCity2000
      RETURN(YM2612_status_port_0_A_r(address));

    case 0x04e0:
    case 0x04e1:
    case 0x04e2:
    case 0x04e3:
      BX_INFO(("volume port read, address=0x%.4x (ignored).\n",
	       (unsigned) address));
      RETURN(0);

    case 0x04e9:
      RETURN((BX_FMSOUND_THIS ym2612_irq ? 0x01 : 0x00) |
	     ((BX_FMSOUND_THIS pcm_int_flag != 0) ? 0x08 : 0x00));

    case 0x04ea:
      RETURN(BX_FMSOUND_THIS pcm_int_mask);

    case 0x04eb:
      ret = BX_FMSOUND_THIS pcm_int_flag;
      BX_FMSOUND_THIS pcm_int_flag = 0;
      BX_FMSOUND_THIS update_irq();
      RETURN(ret);

    default:
      BX_PANIC(("unsupported fmsound read, address=0x%.4x!\n",
        (unsigned) address));
      RETURN(0);
      break;
    }
  read_return:
  if (bx_dbg.fmsound)
    BX_INFO(("FMSOUND read from address: 0x%x = 0x%x\n",
      (unsigned) address, (unsigned) ret));
  return ret;
}
#undef RETURN


  // static IO port write callback handler
  // redirects to non-static class handler to avoid virtual functions

  void
bx_fmsound_c::write_handler(void *this_ptr, Bit32u address, Bit32u value, unsigned io_len)
{
#if !BX_USE_FMSND_SMF
  bx_fmsound_c *class_ptr = (bx_fmsound_c *) this_ptr;

  class_ptr->write(address, value, io_len);
}

  void
bx_fmsound_c::write(Bit32u address, Bit32u value, unsigned io_len)
{
#else
  UNUSED(this_ptr);
#endif  // !BX_USE_FMSND_SMF

  if (io_len > 1)
    BX_PANIC(("fmsound: io write to address %08x len=%u\n",
	      (unsigned) address, (unsigned) io_len));

  if (bx_dbg.fmsound)
    BX_INFO(("FMSOUND write to address: 0x%x = 0x%x\n",
      (unsigned) address, (unsigned) value));

  switch (address) {
    case 0x04d5:
      BX_INFO(("mute port write, address=0x%.4x (ignored).\n",
	       (unsigned) address));
      return;

    case 0x04d8:
      YM2612_control_port_0_A_w(address, value);
      return;

    case 0x04da:
      YM2612_data_port_0_A_w(address, value);
      return;

    case 0x04dc:
      YM2612_control_port_0_B_w(address, value);
      return;

    case 0x04de:
      YM2612_data_port_0_B_w(address, value);
      return;

    case 0x04e0:
    case 0x04e1:
    case 0x04e2:
    case 0x04e3:
      BX_INFO(("volume port write, address=0x%.4x (ignored).\n",
	       (unsigned) address));
      return;

    case 0x04ea:
      BX_FMSOUND_THIS pcm_int_mask = value;
      return;

    case 0x04ec:
      BX_INFO(("LED crtl port write, address=0x%.4x (ignored).\n",
	       (unsigned) address));
      return;

    case 0x04f0:
    case 0x04f1:
    case 0x04f2:
    case 0x04f3:
    case 0x04f4:
    case 0x04f5:
    case 0x04f6:
    case 0x04f7:
    case 0x04f8:
      RF5C68_reg_w(address & 0x000f, value);
      return;

    default:
      BX_PANIC(("unsupported fmsound write, address=0x%.4x!\n",
        (unsigned) address));
      return;
      break;
    }
}

  void
bx_fmsound_c::ym2612_irq_handler(int irq)
{
  bx_fmsound.ym2612_irq = irq;
  bx_fmsound.update_irq();
}

  void
bx_fmsound_c::fm_pcm_interrupt(int bank)
{
  BX_INFO(("bank=%d\n",bank));
  if((BX_FMSOUND_THIS pcm_int_mask&(1<<bank))!=0) {
    BX_FMSOUND_THIS pcm_int_flag|=(1<<bank);
    BX_FMSOUND_THIS update_irq();
  }    
}

void fm_pcm_interrupt(int bank)
{
  bx_fmsound.fm_pcm_interrupt(bank);
}

  void
bx_fmsound_c::update_irq(void)
{
  Boolean newirq;

  newirq=BX_FMSOUND_THIS ym2612_irq || (BX_FMSOUND_THIS pcm_int_flag != 0);

  if(BX_FMSOUND_THIS irq13_raising) {
    if(!newirq) {
      BX_FMSOUND_THIS irq13_raising=0;
      BX_FMSOUND_THIS devices->pic->untrigger_irq(13);
    }
  } else {
    if(newirq) {
      BX_FMSOUND_THIS irq13_raising=1;
      BX_FMSOUND_THIS devices->pic->trigger_irq(13);
    }
  }
}

  Bit8u
bx_fmsound_c::mem_read(Bit32u addr)
{
  return(RF5C68_r(addr&0xfff));
}

  void
bx_fmsound_c::mem_write(Bit32u addr, Bit8u value)
{
  RF5C68_w(addr&0xfff,value);
  return;
}

#endif
